<?php
declare(strict_types=1);

/**
 * CakePHP(tm) : Rapid Development Framework (https://cakephp.org)
 * Copyright (c) Cake Software Foundation, Inc. (https://cakefoundation.org)
 *
 * Licensed under The MIT License
 * For full copyright and license information, please see the LICENSE.txt
 * Redistributions of files must retain the above copyright notice.
 *
 * @copyright Copyright (c) Cake Software Foundation, Inc. (https://cakefoundation.org)
 * @link      https://cakephp.org CakePHP(tm) Project
 * @since     3.3.0
 * @license   https://opensource.org/licenses/mit-license.php MIT License
 */
namespace App;

use Cake\Core\Configure;
use Cake\Core\ContainerInterface;
use Cake\Datasource\FactoryLocator;
use Cake\Error\Middleware\ErrorHandlerMiddleware;
use Cake\Http\BaseApplication;
use Cake\Http\Middleware\BodyParserMiddleware;
use Cake\Http\Middleware\CsrfProtectionMiddleware;
use Cake\Http\MiddlewareQueue;
use Cake\ORM\Locator\TableLocator;
use Cake\Routing\Middleware\AssetMiddleware;
use Cake\Routing\Middleware\RoutingMiddleware;

//authentication
use Authentication\AuthenticationService;
use Authentication\AuthenticationServiceInterface;
use Authentication\AuthenticationServiceProviderInterface;
use Authentication\Identifier\IdentifierInterface;
use Authentication\Middleware\AuthenticationMiddleware;
use Cake\Routing\Router;
use Psr\Http\Message\ServerRequestInterface;

//authorization
use Authorization\AuthorizationService;
use Authorization\AuthorizationServiceInterface;
use Authorization\AuthorizationServiceProviderInterface;
use Authorization\Middleware\AuthorizationMiddleware;
use Authorization\Policy\OrmResolver;
use Psr\Http\Message\ResponseInterface;
use Authorization\Exception\MissingIdentityException;
use Authorization\Exception\ForbiddenException;


/*CSRF protection*/
// For Session based CSRF tokens.
use Cake\Http\Middleware\SessionCsrfProtectionMiddleware;


/**
 * Application setup class.
 *
 * This defines the bootstrapping logic and middleware layers you
 * want to use in your application.
 */
class Application extends BaseApplication implements AuthenticationServiceProviderInterface, AuthorizationServiceProviderInterface
//class Application extends BaseApplication implements AuthorizationServiceProviderInterface
{
    /**
     * Load all the application configuration and bootstrap logic.
     *
     * @return void
     */
    public function bootstrap(): void
    {
        // Call parent to load bootstrap from files.
        parent::bootstrap();

        if (PHP_SAPI === 'cli') {
            $this->bootstrapCli();
        } else {
            FactoryLocator::add(
                'Table',
                (new TableLocator())->allowFallbackClass(false)
            );
        }

        /*
         * Only try to load DebugKit in development mode
         * Debug Kit should not be installed on a production system
         */
        if (Configure::read('debug')) {
            $this->addPlugin('DebugKit');
        }
		
		$this->addPlugin('Authentication');
		$this->addPlugin('Authorization');
		$this->addPlugin('Celestial');
		
        // Load more plugins here
        $this->addPlugin('CakePdf');
        $this->addPlugin('AuditLog');
        $this->addPlugin('AuditStash');
    }

    /**
     * Setup the middleware queue your application will use.
     *
     * @param \Cake\Http\MiddlewareQueue $middlewareQueue The middleware queue to setup.
     * @return \Cake\Http\MiddlewareQueue The updated middleware queue.
     */
    public function middleware(MiddlewareQueue $middlewareQueue): MiddlewareQueue
    {
		
		
        $middlewareQueue
            // Catch any exceptions in the lower layers,
            // and make an error page/response
            ->add(new ErrorHandlerMiddleware(Configure::read('Error')))

            // Handle plugin/theme assets like CakePHP normally does.
            ->add(new AssetMiddleware([
                'cacheTime' => Configure::read('Asset.cacheTime'),
            ]))

            // Add routing middleware.
            // If you have a large number of routes connected, turning on routes
            // caching in production could improve performance. For that when
            // creating the middleware instance specify the cache config name by
            // using it's second constructor argument:
            // `new RoutingMiddleware($this, '_cake_routes_')`
            ->add(new RoutingMiddleware($this))

            // Parse various types of encoded request bodies so that they are
            // available as array through $request->getData()
            // https://book.cakephp.org/4/en/controllers/middleware.html#body-parser-middleware
            ->add(new BodyParserMiddleware())
			
            // Cross Site Request Forgery (CSRF) Protection Middleware
            // https://book.cakephp.org/4/en/security/csrf.html#cross-site-request-forgery-csrf-middleware
            ->add(new CsrfProtectionMiddleware([
                'httponly' => true,
            ]))
			
			// Add the AuthenticationMiddleware. It should be
			// after routing and body parser.
			->add(new AuthenticationMiddleware($this))
			
			//authorization
			//->add(new AuthorizationMiddleware($this));
			->add(new AuthorizationMiddleware($this, [
				'identityDecorator' => function ($auth, $user) {
					return $user->setAuthorization($auth);
				}, 
				'requireAuthorizationCheck' => true,
				'unauthorizedHandler' => [
					//'className' => 'Authorization.Redirect', // <--- see here
					'className' => 'CustomRedirect',
					//'url' => '/users/login',
					 'url' => '/CAKETEST/policy/users/unauthorized',
					'queryParam' => 'redirectUrl',
					'exceptions' => [
						MissingIdentityException::class,
						ForbiddenException::class,
						//UnauthorizedHandler::class,
						OtherException::class,
					],
					'custom_param' => true,
				],
				//'requireAuthorizationCheck' => true
				
			]));


        return $middlewareQueue;
    }

    /**
     * Register application container services.
     *
     * @param \Cake\Core\ContainerInterface $container The Container to update.
     * @return void
     * @link https://book.cakephp.org/4/en/development/dependency-injection.html#dependency-injection
     */
    public function services(ContainerInterface $container): void
    {
    }

    /**
     * Bootstrapping for CLI application.
     *
     * That is when running commands.
     *
     * @return void
     */
    protected function bootstrapCli(): void
    {
        $this->addOptionalPlugin('Cake/Repl');
        $this->addOptionalPlugin('Bake');

        $this->addPlugin('Migrations');

        // Load more plugins here
    }
	
	/**
	 * Returns a service provider instance.
	 *
	 * @param \Psr\Http\Message\ServerRequestInterface $request Request
	 * @return \Authentication\AuthenticationServiceInterface
	 */
	public function getAuthenticationService(ServerRequestInterface $request): AuthenticationServiceInterface
	{
		$service = new AuthenticationService();

		// Define where users should be redirected to when they are not authenticated
		$service->setConfig([
			'unauthenticatedRedirect' => Router::url([
					'prefix' 		=> false,
					'plugin' 		=> null,
					'controller' 	=> 'users',
					'action' 		=> 'login',
			]),
			'queryParam' => 'redirect',
		]);

		$fields = [
			IdentifierInterface::CREDENTIAL_USERNAME => 'username',
			IdentifierInterface::CREDENTIAL_PASSWORD => 'password'
		];
		// Load the authenticators. Session should be first.
		$service->loadAuthenticator('Authentication.Session');
		$service->loadAuthenticator('Authentication.Form', [
			'fields' => $fields,
			'loginUrl' => Router::url([
				'prefix' => false,
				'plugin' => null,
				'controller' => 'users',
				'action' => 'login',
			]),
		]);

		// Load identifiers
		$service->loadIdentifier('Authentication.Password', compact('fields'));

		return $service;
	}
	
	 public function getAuthorizationService(ServerRequestInterface $request): AuthorizationServiceInterface{
		$resolver = new OrmResolver();
		return new AuthorizationService($resolver);
	} 


}
